--[[

.DESCRIPTION
  EZDOK Camera for flight simulator restart script
  Copyright (C) 2014  Radek Henys
  
  The purpose of this script is to kill and start again EZDOK (EZCA) camera utility if it is used
  together with FreeTrack (because this combination makes EZCA crash after some time).
  In simple terms, the script does this:
  
	* Check if FreeTrack and EZCA are running. 
	* Check of EZCA memory usage.
	* Sending a keystroke to simulator.
	* Termination of EZCA process.
	* Start of EZCA process.
	* Setting of EZCA process priority.
	* Setting of EZCA process affinity.
	* Keeping simulator window foreground.
	* Maximization, restoration or switching to full-screen or exclusive full-screen of flight simulator main window.  
  
  For more details, please see this page: http://mouseviator.com/pc-creations/ezca-and-freetrack-restart-script-v-2-0b/
  and/or read the comments in this file.
  
.LICENSE
  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU Lesser General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.

.NOTES
  File Name: ezca_restart.lua
  Author: Radek Henys (admin@mouseviator.com)
  Version: 1.0b
  Last update: 16.4.2014
  License: LGPL v3

]]--
require("mouseviatorhelper")

ipc.log("EZCA Restart Script started...")
ipc.log("MouseviatorHelper.dll , version: "..MVH_VERSION.." , author: "..MVH_AUTHOR..", compile date: "..MVH_BD.." "..MVH_BT)

-- Those options drives the script behaviour
local ScriptOptions = {
   -- will modify EZCA process priority. Set EZCA_PRIORITY variable to desired priority. 
  ["EZCA_MODIFY_PRIORITY"] = 1, 
  -- will modify EZCA process affinity. Set EZCA_AFFINITY variable to desired affinity.
  ["EZCA_MODIFY_AFFINITY"] = 2,
  -- whether to start EZCA even if it is not running already.
  ["EZCA_START_IF_NOT_RUNNING"] = 4,
  -- whether to restart EZCA only if FreeTrack is running. If this is set and you are not using
  -- FreeTrack right now, EZCA will never be restarted
  ["EZCA_RESTART_ONLY_IF_FREETRACK_RUNNING"] = 8,
  -- whether to make simulator window full-screen. This will remove borders and maximize simulator window
  -- (this will also disable ability to move the window) so it will fill the whole screen like when in full-screen
  ["SIM_MAKE_FULLSCREEN"] = 16,
  -- this option will send Alt+Enter key press after EZCA is restarted. This should switch simulator to full-screen
  -- mode. The script tests if the simulator is running in full-screen to prevent from switching back to windowed mode.
  ["SIM_MAKE_EXCLUSIVE_FULLSCREEN"] = 32,
  -- whether to maximize simulator window.
  ["SIM_MAXIMIZE"] = 64,
  -- whether to keep simulator window foreground. EZCA will most likely steal focus from simulator when starting and this
  -- option will try to prevent this by forcing focus to simulator window during the time EZCA is starting.
  ["SIM_KEEP_FOREGROUND"] = 128,
  -- whether to watch EZCA memory usage as trigger to restart EZCA.
  ["EZCA_RESTART_BY_MEM_LIMIT"] = 256,
  -- whether to send key press before EZCA will be killed.
  ["BK_SEND_KEYPRESS"] = 512,
  -- whether to check if EZCA is running on script start. If EZCA is not running, the restart/check function
  -- will NOT be scheduled to run every couple of seconds, so no EZCA restart will done during the simulator session.
  -- Also, the script will be terminated right and so will not use any resources.
  ["CHECK_FREETRACK_ON_START"] = 1024
}

--[[
-- Definition of constants
]]--
-- name of EZCA process
local EZCA_PROCESS_NAME = "EZCA"
-- path to EZCA executable file
-- Edit this to match your installation path 
--(for example replace "Program Files (x86)" with "Program Files" if you do not have 64bit operating system)
local EZCA_EXECUTABLE = "C:\\Program Files (x86)\\EZCA\\EZCA.exe"
-- name of FreeTrack process
local FREETRACK_PROCESS_NAME = "freetrack"
-- how long should be the waiting loop after the EZCA is started again
local EZCA_AFTER_START_WAIT_TIME = 5000
-- the maximum size of EZCA private bytes. If it is "like" that the EZCA private bytes will
-- higher than this value by the next time the restart/check function runs, the script should
-- continue to restart EZCA, otherwise EZCA will not be restarted
local EZCA_PRIVATE_BYTES_LIMIT = 153000
-- priority for EZCA, P_NORMAL is defined in MouseviatorHelper.dll
-- Other possible values are: P_ABOVE_NORMAL, P_BELOW_NORMAL, P_HIGH, P_IDLE and P_REALTIME
local EZCA_PRIORITY = P_NORMAL
-- affinity for EZCA process. This is kind of advanced setting. If do not know how to compute the
-- desired value, keep the EZCA_MODIFY_AFFINITY option disabled.
local EZCA_AFFINITY = 0x1
-- key press to send before EZCA is killed. See FSUIPC for Advanced Users manual for possible values.
-- The default one is "1" as I have this set a "hot key" for virtual cockpit view for most of the planes
local KEYPRESS = 49
-- wait time after key press is send
local AFTER_KEYPRESS_WAIT_TIME = 1000
-- the text in the title bar of the simulator. Partial match is enough.
-- Mine settings is: "Monitor IPC" because I display VAS usage in title bar. 
-- Use "Microsoft Flight" or "Prepar3D" if you have your title bar text unchanged...
local SIMULATOR_WINDOW_TEXT = "Microsoft Flight"
-- how often run the restart/checking function? Value in milliseconds.
local RESTART_CHECK_PERIOD = 180000

-- Set script settings
-- Comment settings you do not want to use
-- Uncomment settings you want to use
-- Write down the settings I forgot to...
local SCRIPT_SETTINGS = ScriptOptions.EZCA_START_IF_NOT_RUNNING
SCRIPT_SETTINGS = logic.Or(SCRIPT_SETTINGS, ScriptOptions.EZCA_RESTART_ONLY_IF_FREETRACK_RUNNING)
SCRIPT_SETTINGS = logic.Or(SCRIPT_SETTINGS, ScriptOptions.SIM_KEEP_FOREGROUND)
--SCRIPT_SETTINGS = logic.Or(SCRIPT_SETTINGS, ScriptOptions.SIM_MAKE_FULLSCREEN)
SCRIPT_SETTINGS = logic.Or(SCRIPT_SETTINGS, ScriptOptions.SIM_MAXIMIZE)
--SCRIPT_SETTINGS = logic.Or(SCRIPT_SETTINGS, ScriptOptions.SIM_MAKE_EXCLUSIVE_FULLSCREEN)
SCRIPT_SETTINGS = logic.Or(SCRIPT_SETTINGS, ScriptOptions.EZCA_RESTART_BY_MEM_LIMIT)
SCRIPT_SETTINGS = logic.Or(SCRIPT_SETTINGS, ScriptOptions.BK_SEND_KEYPRESS)
--SCRIPT_SETTINGS = logic.Or(SCRIPT_SETTINGS, ScriptOptions.EZCA_MODIFY_PRIORITY)
--SCRIPT_SETTINGS = logic.Or(SCRIPT_SETTINGS, ScriptOptions.EZCA_MODIFY_AFFINITY)
SCRIPT_SETTINGS = logic.Or(SCRIPT_SETTINGS, ScriptOptions.CHECK_FREETRACK_ON_START)

-- in this variable we store EZCA private bytes usage
local prev_ezca_pb = 0

-- constants for window style
local WS_CAPTION = 0x00C00000

--[[
Helper function to report errors raised by calling functions from MouseviatorHelper.dll
]]--
function reportProcessLibError(status, value, func_name, args)
  -- if status if false, value will be error message (string)
  if status == false then
    if type(args) == "table" then
      ipc.log("The call of function: "..tostring(func_name).." failed with following error: "..tostring(value))
      ipc.log("The above function was called with following parameters:")
      for k, v in pairs(args) do
        ipc.log("\t"..tostring(k).." -> "..tostring(v))
      end
    else
      ipc.log("The call of function: "..tostring(func_name).."("..tostring(args)..") failed with following error: "..tostring(value))
    end
  end

  return not status
end

--[[
Wrapper function to close system process handle. It will print error if it fails.
]]--
function closehandle(handle)
  if validhandle(handle) then
    status, error_msg = pcall(process.closehandle, handle)
    if status == false then
      ipc.log("Failed to close handle: "..tostring(handle).." due to following error: "..error_msg)
    end
  end
end

--[[
Simple function to test if given handle is valid. We use this to check process handles.
]]--
function validhandle(handle)
  return (handle ~= nil and handle > 0)
end

--[[
Simple function to check if given process is running. Returns nil if not, handle if it does.
]]--
function isprocessrunning(process_name)
  local handle = nil

  if ext.isrunning(process_name) then
    handle = ext.gethandle(process_name)
    if validhandle(handle) then
      ipc.log(process_name.." is running and its handle is: "..tostring(handle))
    else
      -- this should not happen ever
      ipc.log("Weird, ext.isrunning says that: "..process_name.." is running but its handle is zero or negative, right? -> "..tostring(handle))
      handle = nil
    end
  end
  return handle
end

--[[
Function to restart EZCA
]]--
function RestartEZCA()

  -- 1) Test if FreeTrack is running
  local freetrack_handle = isprocessrunning(FREETRACK_PROCESS_NAME)
  if validhandle(freetrack_handle) == false then
    ipc.log("FreeTrack process not found!")
    -- quit, unless we want to continue even when FreeTrack is NOT running
    if logic.And(SCRIPT_SETTINGS, ScriptOptions.EZCA_RESTART_ONLY_IF_FREETRACK_RUNNING) ~= 0 then return end
  end

  -- 2) Test if EZCA is running
  local ezca_handle = isprocessrunning(EZCA_PROCESS_NAME)
  if validhandle(ezca_handle) == false then
    ipc.log("EZCA process not found!")
    -- continue unless we want to start even if EZCA is not running
    if logic.And(SCRIPT_SETTINGS, ScriptOptions.EZCA_START_IF_NOT_RUNNING) == 0 then return end
  end

  -- 3) Now take a look if we should restart by memory limit
  if validhandle(ezca_handle) and logic.And(SCRIPT_SETTINGS, ScriptOptions.EZCA_RESTART_BY_MEM_LIMIT) ~= 0 then
    -- we will need system handle for this
    status, sys_ezca_handle = pcall(process.gethandle, EZCA_PROCESS_NAME)
    if reportProcessLibError(status, sys_ezca_handle, "process.gethandle", EZCA_PROCESS_NAME) == true then return end

    -- initialize previous private bytes
    if prev_ezca_pb <= 0 then
      status, prev_ezca_pb = pcall(process.getprivatebytes, sys_ezca_handle)
      -- quit function if failed to get private bytes
      if reportProcessLibError(status, prev_ezca_pb, "process.getprivatebytes", sys_ezca_handle) == true then
        closehandle(sys_ezca_handle)
        return
      end
      prev_ezca_pb = prev_ezca_pb / 1024 / 2
    end

    -- get current EZCA private bytes
    status, curr_ezca_pb = pcall(process.getprivatebytes, sys_ezca_handle)
    if reportProcessLibError(status, curr_ezca_pb, "process.getprivatebytes", sys_ezca_handle) == true then
      closehandle(sys_ezca_handle)
      return
    end
    curr_ezca_pb = curr_ezca_pb / 1024

    -- calculate private bytes difference
    pb_diff = curr_ezca_pb - prev_ezca_pb
    -- calculate future private bytes
    future_pb = curr_ezca_pb + pb_diff
    -- test if next time function runs the EZCA private bytes will overcome limit or NOT
    if future_pb >= EZCA_PRIVATE_BYTES_LIMIT then
      ipc.log("EZCA private bytes would be around: "..future_pb.." by next time the script runs. That is more than safety value of: "..EZCA_PRIVATE_BYTES_LIMIT.." .Continuing to restart EZCA...")
      -- We will restart EZCA, so reset previous private bytes counter
      prev_ezca_pb = 0
    else
      ipc.log("EZCA private bytes would be around: "..future_pb.." by next time the script runs. That is less than safety value of: "..EZCA_PRIVATE_BYTES_LIMIT.." . Will wait for next execution. Bye")
      prev_ezca_pb = curr_ezca_pb
      closehandle(sys_ezca_handle)
      return
    end

    --close system EZCA handle
    closehandle(sys_ezca_handle)
    sys_ezca_handle = nil
  end

  -- 4) send key press before killing EZCA
  -- this is good to restore view to default before EZCA closes
  if logic.And(SCRIPT_SETTINGS, ScriptOptions.BK_SEND_KEYPRESS) ~= 0 then
	ipc.log("Sending key press: "..tostring(KEYPRESS))
	-- make sure simulator has focus, or the key press will not be caught
	ext.focus()	
	ipc.sleep(100)
    ipc.keypress(KEYPRESS)
    -- adjust this sleep value if the view does not have enough time to move
    ipc.sleep(AFTER_KEYPRESS_WAIT_TIME)
  end

  --[[ Here:
  * FreeTrack is running or EZCA_RESTART_ONLY_IF_FREETRACK_RUNNING is not set
  * EZCA is running or EZCA is not running and EZCA_START_IF_NOT_RUNNING is set
  * If EZCA_RESTART_BY_MEM_LIMIT is set, the limit was reached
  --]]
  --Kill EZCA if running
  if validhandle(ezca_handle) then
    -- kill EZCA
    ipc.log("Killing EZCA (handle: "..ezca_handle..")")
    ext.kill(ezca_handle)
    ipc.sleep(500)
  end

  ipc.log("Trying to run EZCA...")
  ezca_handle, error_msg = ext.run(EZCA_EXECUTABLE, EXT_HIDE)
  if validhandle(ezca_handle) then
    ipc.log("EZCA started again. Handle is: "..ezca_handle)
  else
    ipc.log("Failed to start EZCA from: "..EZCA_EXECUTABLE.." ,due to following error:"..error_msg)
    return
  end

  ipc.log("Starting EZCA wait loop...")

  --are we going to need system EZCA handle?
  --The handle we get from ext.gethandle is FSUIPC specific, we cannot use it with MouseviatorHelper library
  if logic.And(SCRIPT_SETTINGS, ScriptOptions.EZCA_MODIFY_PRIORITY) ~= 0 or
    logic.And(SCRIPT_SETTINGS, ScriptOptions.EZCA_MODIFY_AFFINITY) ~= 0 then
    -- get EZCA system handle
    status, sys_ezca_handle = pcall(process.gethandle, EZCA_PROCESS_NAME)
    reportProcessLibError(status, sys_ezca_handle, "process.gethandle", EZCA_PROCESS_NAME)
    ipc.log("EZCA system handle:"..sys_ezca_handle)
  end

  local start_time = ipc.elapsedtime()
  repeat

    -- check whether to modify priority
    if logic.And(SCRIPT_SETTINGS, ScriptOptions.EZCA_MODIFY_PRIORITY) ~= 0 and validhandle(sys_ezca_handle) then
      -- and modify priority
      status, error_msg = pcall(process.setpriority, sys_ezca_handle, EZCA_PRIORITY)
      reportProcessLibError(status, error_msg, "process.setpriority", EZCA_PRIORITY)
    end

    -- check whether to modify affinity
    if logic.And(SCRIPT_SETTINGS, ScriptOptions.EZCA_MODIFY_AFFINITY) ~= 0 and validhandle(sys_ezca_handle) then
      -- and modify priority
      status, error_msg = pcall(process.setaffinity, sys_ezca_handle, EZCA_AFFINITY)
      reportProcessLibError(status, error_msg, "process.setaffinity", EZCA_AFFINITY)
    end

    -- check whether to keep focus on FS
    if logic.And(SCRIPT_SETTINGS, ScriptOptions.SIM_KEEP_FOREGROUND) ~= 0 then
      ipc.log("Trying to return focus to FS...")
      ext.focus()
    end
    -- sleep for a while
    ipc.sleep(500)
  until (ipc.elapsedtime() - start_time) > EZCA_AFTER_START_WAIT_TIME

  ipc.log("EZCA wait loop finished!")

  -- close system EZCA handle
  if sys_ezca_handle ~= nil and sys_ezca_handle > 0 then
    closehandle(sys_ezca_handle)
    sys_ezca_handle = nil
  end

  -- check whether to manipulate with simulator window
  if logic.And(SCRIPT_SETTINGS, ScriptOptions.SIM_MAXIMIZE) ~= 0 or
    logic.And(SCRIPT_SETTINGS, ScriptOptions.SIM_MAKE_FULLSCREEN) ~= 0 or
    logic.And(SCRIPT_SETTINGS, ScriptOptions.SIM_MAKE_EXCLUSIVE_FULLSCREEN) ~= 0 then

    -- get flight simulator process id
    status, pid = pcall(process.getid)
    if reportProcessLibError(status, pid, "process.getid", 0) == true then return end
    
    -- we need to get handle to main simulator window
    -- function call below finds window with given text, it must also belong to process with id = pid and must
    -- be top-level window (no child window)
    status, whandles = pcall(window.find, SIMULATOR_WINDOW_TEXT,pid,WE_TOPLEVELONLY)
    if reportProcessLibError(status, whandles, "window.find", SIMULATOR_WINDOW_TEXT) == true then return end

    -- handle to main simulator window will be at whandles[1], there is no check to verify this
    if logic.And(SCRIPT_SETTINGS, ScriptOptions.SIM_MAKE_EXCLUSIVE_FULLSCREEN) ~= 0 then
      -- get window style
      status, style = pcall(window.getstyle, whandles[1])
      if reportProcessLibError(status, style, "window.find", whandles[1]) == true then return end
      -- check if window is full-screen
      status, fullscreen = pcall(window.isfullscreen, whandles[1])
      if reportProcessLibError(status, fullscreen, "window.isfullscreen", whandles[1]) == true then return end

	  -- has the window caption? if not. it is in full-screen mode
      hasCaption = logic.And(style, WS_CAPTION) > 0
	  ipc.log("Has Caption: "..tostring(hasCaption)..", full-screen: "..tostring(fullscreen))
      if hasCaption and not fullscreen then
        -- this sends Alt+Enter, the default key combination to switch to full-screen and back
        ipc.log("Trying to switch simulator to full-screen mode using default keystroke...")
        ipc.keypress(13,16)
      else
        ipc.log("Simulator seems to be running full-screen already! (Window does not have caption, window.isfullscreen returned: "..tostring(fullscreen).."). Restoring window...")
		-- Now the tricky part. Prepar3D v2.1 does not redraw window correctly in full-screen if we call window.restore, so we will not call that if 
		-- sim version is 10=P3D, but this means Prepar3D v1.4, which is like FSX
		-- FSX on the other hand will not restore correctly using ext.focus() if it was in full-screen and was minimized due to EZCA restart. It needs window.restore for correct restore
		sim_version = ipc.readUW(0x3308)
		-- to make it work with Prepar3D v1.4, you might need to comment if-else-end below and leave just the two commands between if-else
		if sim_version == 8 then
			status, error_msg = pcall(window.restore, whandles[1])
			reportProcessLibError(status, error_msg, "window.restore", whandles[1])
		else
			ipc.log("Unsupported simulator version: "..tostring(sim_version)..".Simulator window might NOT have restore correctly if it was minimized due to EZCA restart...") 
		end
	  end
    elseif logic.And(SCRIPT_SETTINGS, ScriptOptions.SIM_MAKE_FULLSCREEN) ~= 0 then
      -- there should be no issue calling this on window that is already a fullscreen window
      ipc.log("Trying to switch simulator to full-screen window mode..")
      status, error_msg = pcall(window.setfullscreen, whandles[1])
      reportProcessLibError(status, error_msg, "window.setfullscreen", whandles[1])
    else
      -- maximize fs window
      ipc.log("Trying to maximize simulator window (Handle: "..tostring(whandles[1])..")...")
      status, error_msg = pcall(window.maximize, whandles[1])
      reportProcessLibError(status, error_msg, "window.maximize", whandles[1])
    end
    -- give simulator focus
    ext.focus()
  end
end

-- should we bother if freetrack is not running at all?
if logic.And(SCRIPT_SETTINGS, ScriptOptions.CHECK_FREETRACK_ON_START) ~= 0 then
  local freetrack_handle = isprocessrunning(FREETRACK_PROCESS_NAME)
  if validhandle(freetrack_handle) == false then
    ipc.log("FreeTrack process not found! The script settings tells me not to bother than. No more checks will be done for this simulator session!")
    -- quit
    ipc.log("EZCA Restart Script end...")
    ipc.exit()
  end
end

-- call our restart function by a timer
ipc.log("Setting EZCA restart function to be called every: "..tostring(RESTART_CHECK_PERIOD / 1000).." seconds")
event.timer(RESTART_CHECK_PERIOD, "RestartEZCA")

-- uncomment the line below if you want to perform check when simulator loads
--RestartEZCA()

ipc.log("EZCA Restart Script end...")